<?php
require_once '../config.php';
requireLogin();

// Get pending orders
try {
    $stmt = $pdo->query("
        SELECT o.*, pg.name as gateway_name 
        FROM orders o 
        LEFT JOIN payment_gateways pg ON o.payment_gateway_id = pg.id
        WHERE o.payment_status = 'pending' 
        ORDER BY o.created_at DESC
    ");
    $orders = $stmt->fetchAll();
} catch(PDOException $e) {
    $orders = [];
}

// Handle order actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $orderId = intval($_POST['order_id'] ?? 0);
    $action = $_POST['action'];
    
    try {
        if ($action === 'approve') {
            $stmt = $pdo->prepare("UPDATE orders SET payment_status = 'completed' WHERE id = ?");
            $stmt->execute([$orderId]);
            setFlash('success', 'Order approved successfully');
        } elseif ($action === 'reject') {
            $stmt = $pdo->prepare("UPDATE orders SET payment_status = 'failed' WHERE id = ?");
            $stmt->execute([$orderId]);
            setFlash('success', 'Order rejected');
        }
        header('Location: pending-orders.php');
        exit;
    } catch(PDOException $e) {
        setFlash('error', 'Action failed');
    }
}

$flash = getFlash();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pending Orders - Prime Institute Admin</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="admin-style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>
        
        <main class="main-content">
            <div class="page-header">
                <h1>⏳ Pending Orders</h1>
                <p>Orders waiting for payment confirmation</p>
            </div>
            
            <?php if ($flash): ?>
            <div class="alert alert-<?php echo $flash['type']; ?>">
                <?php echo htmlspecialchars($flash['message']); ?>
            </div>
            <?php endif; ?>
            
            <div class="content-card">
                <div class="table-container">
                    <?php if (empty($orders)): ?>
                        <p style="text-align: center; padding: 40px; color: #9ca3af;">No pending orders</p>
                    <?php else: ?>
                    <table>
                        <thead>
                            <tr>
                                <th>Order ID</th>
                                <th>Customer</th>
                                <th>Email</th>
                                <th>Phone</th>
                                <th>Amount</th>
                                <th>Gateway</th>
                                <th>Transaction ID</th>
                                <th>Payment Proof</th>
                                <th>Date</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($orders as $order): ?>
                            <tr>
                                <td><strong><?php echo htmlspecialchars($order['order_id']); ?></strong></td>
                                <td><?php echo htmlspecialchars($order['name']); ?></td>
                                <td><?php echo htmlspecialchars($order['email']); ?></td>
                                <td><?php echo htmlspecialchars($order['phone']); ?></td>
                                <td>৳<?php echo number_format($order['amount'], 0); ?></td>
                                <td><?php echo htmlspecialchars($order['gateway_name'] ?? 'N/A'); ?></td>
                                <td><?php echo htmlspecialchars($order['transaction_id'] ?? '-'); ?></td>
                                <td>
                                    <?php if (!empty($order['payment_proof'])): ?>
                                        <a href="../uploads/<?php echo htmlspecialchars($order['payment_proof']); ?>" 
                                           target="_blank" class="btn btn-sm" 
                                           style="background: #3b82f6; color: white;" 
                                           title="View Screenshot">
                                            📷 View
                                        </a>
                                        <button type="button" class="btn btn-sm" 
                                                style="background: #8b5cf6; color: white;" 
                                                onclick="showImageModal('../uploads/<?php echo htmlspecialchars($order['payment_proof']); ?>')">
                                            🔍 Preview
                                        </button>
                                    <?php else: ?>
                                        <span style="color: #9ca3af;">No proof</span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo date('d M Y, h:i A', strtotime($order['created_at'])); ?></td>
                                <td>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                        <button type="submit" name="action" value="approve" class="btn btn-success btn-sm" 
                                                onclick="return confirm('Approve this order?')">✓ Approve</button>
                                        <button type="submit" name="action" value="reject" class="btn btn-danger btn-sm" 
                                                onclick="return confirm('Reject this order?')">✗ Reject</button>
                                    </form>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
    
    <!-- Image Modal -->
    <div id="imageModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.9); z-index: 10000; justify-content: center; align-items: center;">
        <span onclick="closeImageModal()" style="position: absolute; top: 20px; right: 30px; color: white; font-size: 40px; cursor: pointer;">&times;</span>
        <img id="modalImage" style="max-width: 90%; max-height: 90%; border-radius: 8px;">
    </div>
    
    <script>
        function showImageModal(src) {
            document.getElementById('modalImage').src = src;
            document.getElementById('imageModal').style.display = 'flex';
        }
        
        function closeImageModal() {
            document.getElementById('imageModal').style.display = 'none';
        }
        
        document.getElementById('imageModal').addEventListener('click', function(e) {
            if (e.target === this) closeImageModal();
        });
        
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') closeImageModal();
        });
    </script>
</body>
</html>
